#!/opt/alt/python37/bin/python3.7 -bb
# -*- coding: utf-8 -*-

# Copyright © Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2019 All Rights Reserved
#
# Licensed under CLOUD LINUX LICENSE AGREEMENT
# http://cloudlinux.com/docs/LICENSE.TXT

from __future__ import print_function
from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals
from future import standard_library
standard_library.install_aliases()
from builtins import *
import glob
import os
import sys
from pwd import getpwnam

from clcommon.utils import get_file_lines, write_file_lines
from clcommon.cpapi import cpusers, NotSupported
from clcommon.utils import mod_makedirs

# cagefs imports
LIBDIR = '/usr/share/cagefs'
sys.path.append(LIBDIR)
from cagefsctl import EXCLUDE_PATH

NEW_EXCLUDE_PATH = '/usr/share/cagefs/exclude.d'


def _is_user_present_in_system(username):
    try:
        getpwnam(username)
        return True
    except KeyError:
        return False


def _is_exclude_user(cp_users, username):
    """
    Determines whether the user should be excluded from CageFS
    :param cp_users: Panel users lst
    :param username: username to check
    :return: True - user should be excluded from CageFs, False - else
    """
    if username in cp_users:
        # Panel user should not be excluded from CageFS
        return False
    if _is_user_present_in_system(username):
        # User present in system, but absent in panel - exclude from CageFS
        return True
    return False


def _process_file(cp_users, source_filename, dest_filename):
    """
    Copies usernames from source file to destination file and excudes absent and panel users
    :param source_filename: Source filename (/usr/share/cagefs/exclude.d/..)
    :param dest_filename: Destination filename (/etc/cagefs/exclude/..)
    :return: None
    """
    source_lines = get_file_lines(source_filename)
    if os.path.isfile(dest_filename):
        dest_lines = get_file_lines(dest_filename)
    else:
        dest_lines = []
    # Determine manually added users to destination file and put them to output line list
    source_users = [x.strip() for x in source_lines]
    dest_users = [x.strip() for x in dest_lines]
    manually_added_users_list = list(set(dest_users) - set(source_users))
    lines_for_write = [line+'\n' for line in manually_added_users_list if line]
    for username in source_users:
        if _is_exclude_user(cp_users, username):
            lines_for_write.append(username + '\n')
    # Write new file if need
    if not lines_for_write:
        # Nothing to write
        try:
            os.remove(dest_filename)
        except (OSError, IOError):
            pass
    else:
        if sorted(dest_lines) != sorted(lines_for_write):
            write_file_lines(dest_filename, lines_for_write, 'w')
            os.chmod(dest_filename, 0o0600)


def main():
    # Create /etc/cagefs/exclude directory
    try:
        if not os.path.isdir(EXCLUDE_PATH):
            mod_makedirs(EXCLUDE_PATH, 0o0751)
    except (OSError, IOError) as e:
        print("Error:", str(e))
        sys.exit(1)
    try:
        # Get panel users list
        cp_users = cpusers()
    except NotSupported:
        cp_users = ()
    exclude_file_list = glob.glob(os.path.join(NEW_EXCLUDE_PATH, '*'))
    is_error = False
    for exclude_file in exclude_file_list:
        exclude_file_to_write = os.path.join(EXCLUDE_PATH, os.path.basename(exclude_file))
        try:
            _process_file(cp_users, exclude_file, exclude_file_to_write)
        except (OSError, IOError) as e:
            print("Error:", str(e))
            is_error = True
    sys.exit(1 if is_error else 0)


if "__main__" == __name__:
    main()
